
#include "dvlbutton_mod.h"

#define MODULE_NAME  "dvlbutton"
MODULE_LICENSE("GPL");

static unsigned int longtime  = 5000;
static unsigned int shorttime = 1000;

static struct dvlbutton_context *global_contexts = NULL;

static unsigned int dvlbutton_major = 245; // 0=use automatic majornumber, not static
                                           // can be read from /proc/devices
static unsigned int dvlbutton_minor = 0;   // use automatic minornumber

static dev_t dev_id;
static struct dvlbutton_dev global_dev;

//
// Helper functions
//
 
static int num_buttons(void)
{
	int rv = 0;
	struct dvlbutton_entry *b;
	
	for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
	{
	}
	
	return rv;
}

static int button_idx(char id)
{
    int rv = 0;
    struct dvlbutton_entry *b;
    for (b = global_buttons, rv = 0; b->type != TYPE_NONE; b++, rv++)
    {
        if (b->type == TYPE_BUTTON && b->c1 == id) return rv;
    }
    return -1;
}

static unsigned long timespec_diff(const struct timespec *t1, const struct timespec *t2)
{
	unsigned long s1 = t1->tv_sec;
	unsigned long s2 = t2->tv_sec;
	unsigned long m1 = t1->tv_nsec / 1000000L;
	unsigned long m2 = t2->tv_nsec / 1000000L;

	unsigned long md = (m2 + (s2 - s1) * 1000L) - m1;
	return md;
}

#define BOUNCE_TIME_MS 20
static int in_bounce_time(const struct timespec *t1, const struct timespec *t2)
{
	unsigned long md = timespec_diff(t1, t2);

	if (md <= BOUNCE_TIME_MS)
		return 1;
	return 0;
}

//
// Output to the character device
//

static char output_queue[64];
static unsigned int write_pos = 0;
static unsigned int file_offset = 0;

static void move_queue(void)
{
	unsigned int offset = sizeof(output_queue) / 2;
	unsigned int i;
	
	for (i = 0; i < sizeof(output_queue) - offset; i++)
		output_queue[i] = output_queue[i + offset];
		
	write_pos -= offset;
	file_offset += offset;
}

static void enqueue_char(struct dvlbutton_dev *dev, char c)
{
	output_queue[write_pos++] = c;
	if (write_pos >= sizeof(output_queue))
		move_queue();
	wake_up_interruptible_all(&dev->inq);
}
 
static ssize_t emit_char(char __user *buf, unsigned int offset)
{
	ssize_t retval = 0;

	if (copy_to_user(buf, &(output_queue[offset]), sizeof(char)))
		retval = -EFAULT;
	else
		retval = (ssize_t) sizeof(char);
    
	return retval;
}

#define CHARS_PER_BUTTON 128
static ssize_t emit_states(char __user *buf)
{
	ssize_t retval = 0;
	int i;
	int num = num_buttons();
	
	char *s = (char *) kzalloc((num + 1) * CHARS_PER_BUTTON, GFP_KERNEL);
	if (s == NULL)
	{
		printk(KERN_ERR "unable to allocate memory\n");
		return retval;
	}

	for (i = 0; i < num; i++)
	{
		char line[CHARS_PER_BUTTON + 1];
		char c = DVLBUTTON_NONE;
		struct dvlbutton_entry *b = &(global_buttons[i]);
		
		int value = 0;
		if (b->gpio >= 0)
			dvl_gpio_line_get(b->gpio, &value);
		
		value = !!value;
		if (b->polarity)
			value = !value;
			
		if (b->type == TYPE_BUTTON || b->type == TYPE_BUTTON_SHORTLONG)
		{
			c = value ? (b->c1) : DVLBUTTON_NONE;
		}
		else if (b->type == TYPE_SWITCH)
		{
			c = value ? (b->c1) : (b->c2);
		}
		
		snprintf(line, sizeof(line), "%c - %s\n", c, b->name);
		strcat(s, line);
	}
	
	if (copy_to_user(buf, s, strlen(s)))
		retval = -EFAULT;
	else
		retval = (ssize_t) strlen(s);
	
	kfree(s);
	return retval;
}

//
// Timers and IRQ handlers
// 

/*
 * Called if timer is expired
 */
static void timer_handler(unsigned long arg) {
	int button = (int) arg;
    global_contexts[button].is_long = 1; /* remember until release of button */
   	enqueue_char(global_contexts[button].dev, global_buttons[button].c2);
}


/*
 * Called if button is released
 */
static irqreturn_t shortlong_stop_handler(int button, void *unused_ctx)
{
	del_timer_sync(&(global_contexts[button].timer));
	if (global_contexts[button].is_long == 0)
	{
		struct timespec now;
		ktime_get_ts(&now);
		
		if (timespec_diff(&(global_contexts[button].last_time), &now) <= shorttime)
		{
			enqueue_char(global_contexts[button].dev, global_buttons[button].c1);
		}
	}
	else
	{
		global_contexts[button].is_long = 0;
	}
	return IRQ_HANDLED;
}

/*
 * Called if button is pressed
 */
static irqreturn_t shortlong_start_handler(int button, void *unused_ctx) {
	init_timer(&(global_contexts[button].timer));
	global_contexts[button].timer.data = (unsigned long) button;
	global_contexts[button].timer.function = timer_handler;
	global_contexts[button].timer.expires = jiffies + longtime * (unsigned int) HZ / 1000;
	add_timer(&(global_contexts[button].timer));
	return IRQ_HANDLED;
}

/*
 * Called if button is released or pressed
 */
static irqreturn_t irq_handler(int irq, void* ctx) 
{
	irqreturn_t rv = IRQ_HANDLED;
	int button = -1, change = -1;
	struct timespec now;

	dvl_gpio_change_get(irq, &button, &change);
	dvl_gpio_clear();

	if (button >= 0)
	{
		struct dvlbutton_entry *b = &(global_buttons[button]);
		ktime_get_ts(&now);

		change = !!change;
		if (b->polarity)
			change = !change;
		
		if (change != global_contexts[button].last_change)
		{
			if (!in_bounce_time(&(global_contexts[button].last_time), &now))
			{
				if (b->type == TYPE_BUTTON)
				{
					if (change) // button pressed
					{
					}
					else // button released
					{
						enqueue_char(global_contexts[button].dev, b->c1);
					}
				}
				else if (b->type == TYPE_BUTTON_SHORTLONG)
				{
					if (change) // button pressed
					{
						rv = shortlong_start_handler(button, ctx);
					}
					else // button released
					{
						rv = shortlong_stop_handler(button, ctx);
					}
				}
				else if (b->type == TYPE_SWITCH)
				{
					if (change) // switch enabled
					{
						enqueue_char(global_contexts[button].dev, b->c1);
					}
					else // switch disabled
					{
						enqueue_char(global_contexts[button].dev, b->c2);
					}
				}
				
				global_contexts[button].last_change = change;
				memcpy(&(global_contexts[button].last_time), &now, sizeof(struct timespec));
			}
		}
	}
	return rv;
}

//
// dvlbutton
// 

/*
 * This function is called in select(), poll() etc
 */
static unsigned int dvlbutton_poll(struct file *filp, poll_table *wait) {
    struct dvlbutton_dev *dev = filp->private_data;
    unsigned int mask = 0;

	unsigned int read_pos;
	if ((unsigned int) (filp->f_pos) < file_offset)
	{
		printk(KERN_WARNING "Requested button events are not available anymore\n");
		filp->f_pos = file_offset;
	}
	read_pos = (unsigned int) (filp->f_pos) - file_offset;

    down(&dev->sem);
    poll_wait(filp, &dev->inq, wait);
    if (read_pos < write_pos)
        mask |= POLLIN | POLLRDNORM; /* readable */

    up(&dev->sem);
    return mask;
}

/*
 * Called if read() is called
 */
static ssize_t dvlbutton_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos) {
    struct dvlbutton_dev *dev = filp->private_data;
    ssize_t retval = 0;

	unsigned int read_pos;
	if ((unsigned int) (filp->f_pos) < file_offset)
	{
		printk(KERN_WARNING "Requested button events are not available anymore\n");
		filp->f_pos = file_offset;
	}
	read_pos = (unsigned int) (filp->f_pos) - file_offset;

    if (down_interruptible(&dev->sem))
        return -ERESTARTSYS;
        
	while (!(filp->f_flags & O_NONBLOCK) && (read_pos >= write_pos)) { /* nothing to read */
	    up(&dev->sem);
	    if (wait_event_interruptible(dev->inq, read_pos < write_pos))
	        return -ERESTARTSYS;
	    if (down_interruptible(&dev->sem))
	        return -ERESTARTSYS;
	}

	if(!(filp->f_flags & O_NONBLOCK))
    {
		if (count >= 1)
			retval = emit_char(buf, read_pos);
		*f_pos += retval;
	}
	else
	{
		if (count >= num_buttons())
			retval = emit_states(buf);
	}
	
    up(&dev->sem);
    return retval;

}

static int dvlbutton_release(struct inode *unused_inode, struct file* unused_filp) {
    return 0;
}

/*
 * Called if open() is called
 */
static int dvlbutton_open(struct inode *inode, struct file* filp) {
    struct dvlbutton_dev *dev;

    dev = container_of(inode->i_cdev, struct dvlbutton_dev, cdev);
    filp->private_data = dev;
    filp->f_pos = write_pos + file_offset;
    return 0;
}

/*
 * Called if ioctl() is called
 */
static int dvlbutton_ioctl(struct inode *unused_inode, struct file *unused_filp, unsigned int cmd, unsigned long arg) {
    int retval = 0;

    switch (cmd) {
    case DVLBUTTON_IOCTFACDEFTIME: {
        longtime = arg;
    }
        break;
    case DVLBUTTON_IOCQFACDEFTIME: {
        retval = (int) longtime;
    }
        break;
    case DVLBUTTON_IOCRESET: {
    	int i;
    	for (i = 0; i < num_buttons(); i++)
    		global_contexts[i].is_long = 0;
    	
        longtime = 5000;
    }
        break;
    case DVLBUTTON_IOCGBTNSTATE: {
        int idx = button_idx(arg);
        if (idx >= 0 && global_contexts[idx].last_change == 1) retval = 1;
        else if (idx < 0) retval = -1;
    }
        break;
    default:
        return -ENOTTY;
    }
    return retval;
}

static struct file_operations dvlbutton_fops = {
    .owner = THIS_MODULE,
    .open = dvlbutton_open,
    .release = dvlbutton_release,
    .read = dvlbutton_read,
    .poll = dvlbutton_poll,
    .ioctl = dvlbutton_ioctl,
};

/*
 * Setup character devices
 */
static void dvlbutton_setup_cdev(struct dvlbutton_dev *dev, unsigned int index) {
    int err;
    unsigned int devno = MKDEV(dvlbutton_major, dvlbutton_minor + index);

    cdev_init(&dev->cdev, &dvlbutton_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &dvlbutton_fops;
    err = cdev_add(&dev->cdev, devno, 1);
    init_MUTEX(&dev->sem);
    init_waitqueue_head(&dev->inq);
    if (err)
        printk(KERN_DEBUG "Error %d adding %s%d",err,MODULE_NAME,index);
}

/*
 * GPIOs
 */

static void dvl_init_all_gpios(void)
{
	struct dvlbutton_entry *b;
	
	dvl_gpio_init();

	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		if (b->gpio >= 0)
		{
			dvl_gpio_line_config_in(b->gpio);
			printk(KERN_INFO "GPIO %d configured for input\n", b->gpio);
		}
	}
}

/*
 * IRQs
 */

static int dvl_init_all_irqs(void)
{
	int rc = 0;
	struct dvlbutton_entry *b, *c;
	
	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		int irq = b->irq;
		for (c = global_buttons; c != b; c++)
		{
			if (c->irq == b->irq)
				irq = -1;
		}
		
		if (irq >= 0)
		{
			rc = request_irq(
				(unsigned int) irq,
				irq_handler,
				SA_INTERRUPT,
				"devolo button handler",
				&global_dev
			);
			
			printk(KERN_INFO "IRQ %d requested, %s\n", irq, rc < 0 ? "failed" : "successful");
			if (rc < 0)
				return rc;
		}
	}
	
	return rc;
}

static void dvl_shutdown_all_irqs(void)
{
	struct dvlbutton_entry *b, *c;
	
	for (b = global_buttons; b->type != TYPE_NONE; b++)
	{
		int irq = b->irq;
		for (c = global_buttons; c != b; c++)
		{
			if (c->irq == b->irq)
				irq = -1;
		}
		
		if (irq >= 0)
		{
			free_irq((unsigned int) irq, &global_dev);
			printk(KERN_INFO "IRQ %d freed\n", irq);
		}
	}
}

/*
 * Init module after insmod or modprobe
 */
static int __init init_mod(void) {
    int result = 0;
    int num = 0;
    int i;

    dev_id = MKDEV(dvlbutton_major, dvlbutton_minor);
    result = register_chrdev_region(dev_id,1,MODULE_NAME);
    if (result < 0) {
        printk(KERN_WARNING "%s, can't get major %d\n",MODULE_NAME, dvlbutton_major);
        return result;
    }
    dvlbutton_setup_cdev(&global_dev, 0);

	num = num_buttons();
	global_contexts = kcalloc((unsigned int) num, sizeof(struct dvlbutton_context), GFP_KERNEL);
	if (global_contexts == NULL)
	{
		printk(KERN_ERR "unable to allocate memory\n");
		return -ENOMEM;
	}
	for (i = 0; i < num; i++)
	{
		global_contexts[i].last_change = -1;
		ktime_get_ts(&(global_contexts[i].last_time));
		global_contexts[i].dev = &global_dev;
	}

    /* enable the factory default gpio */
    dvl_init_all_gpios();
    
	if (dvl_init_all_irqs() < 0)
	{
		printk(KERN_DEBUG "Factory default button IRQ not available\n"); 
		cdev_del(&global_dev.cdev);
		unregister_chrdev_region(dev_id,1);
		return -EIO;
	}
	dvl_gpio_clear();
    
    printk(KERN_INFO "Module %s initialized\n", MODULE_NAME);   
    return 0;
}

/*
 * Cleanup after rmmod
 */
static void __exit exit_mod(void) {
	int i;
	
    dvl_shutdown_all_irqs();
    cdev_del(&global_dev.cdev);
    unregister_chrdev_region(dev_id,1);
    
    for (i = 0; i < num_buttons(); i++)
    	del_timer_sync(&(global_contexts[i].timer));
    
    if (global_contexts != NULL)
    	kfree(global_contexts);
    
    printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Taedcke / Christian Petry");
MODULE_DESCRIPTION("Button module");

module_param(longtime,uint,S_IRUGO);
module_param(dvlbutton_major,uint,S_IRUGO);
module_param(dvlbutton_minor,uint,S_IRUGO);
module_init(init_mod);
module_exit(exit_mod);
