#include <linux/if_vlan.h>
#include <asm/rt2880/rt_mmap.h>

#include "dvl_led_ctrl.h"

#define _ESW_REG(x)		(*((volatile u32 *)(RALINK_ETH_SW_BASE + x)))
#define R_POA			(0x80)	/* Port Ability */
#define B_P0_LINK		25
#define B_P1_LINK		26
#define B_P2_LINK		27
#define B_P3_LINK		28
#define B_P4_LINK		29
#define B_G0_LINK		30
#define B_G1_LINK		31
#define R_SGC			(0x9C)	/* Switch Global Control */
#define B_LED_FLASH_TIME	23
#define LED_FLASH_TIME_MASK	(0x01800000)
enum { E_30MS, E_60MS, E_240MS, E_480MS };
#define R_P0_LED		(0xA4)	/* Port 0 Led */
#define R_P1_LED		(0xA8)	/* Port 1 Led */
#define R_P2_LED		(0xAC)	/* Port 2 Led */
#define R_P3_LED		(0xB0)	/* Port 3 Led */
#define R_P4_LED		(0xB4)	/* Port 4 Led */
enum { E_LINK, E_100M_SPEED, E_DUPLEX, E_ACTIVITY, E_COLLISION, E_LINK_ACTIVITY, E_DUPLEX_COLLISION, E_10M_SPEED_ACTIVITY, E_100M_SPEED_ACTIVITY };
#define R_PPC			(0xE0)	/* Switch <-> frame engine (port 6) packet counter */
#define R_P0PC			(0xE8)	/* Port 0 packet counter */
#define R_P1PC			(0xEC)	/* Port 1 packet counter */
#define R_P2PC			(0xF0)	/* Port 2 packet counter */
#define R_P3PC			(0xF4)	/* Port 3 packet counter */
#define R_P4PC			(0xF8)	/* Port 4 packet counter */
#define R_P5PC			(0xFC)	/* Port 5 packet counter */
#define PC_MASK			(0xFFFF)

static const unsigned int cLedActiveJiffies 	=  80*HZ/1000;	/*  80 ms */
static const unsigned int cLedInactiveJiffies 	=  40*HZ/1000;	/*  40 ms */
static const unsigned int cSwPktCntPollJiffies	= 200*HZ/1000;	/* 200 ms */

static const unsigned char portsInUse[] = { 0, 2, 4 };
static const unsigned char nPortsInUse = sizeof(portsInUse) / sizeof(portsInUse[0]);
static unsigned short pktCnt[sizeof(portsInUse)/sizeof(portsInUse[0])] = { 0 };
static volatile u32* pktCntAdr[sizeof(portsInUse)/sizeof(portsInUse[0])] = { NULL }; 

struct tasklet_struct dvl_link_tasklet;
static struct timer_list dvlEthLedBlinkTimer;
static struct timer_list dvlEthLedSwPktCntPollTimer;
static int blinkTimerState;
static bool switchActivity;
static bool cpuActivity;
static volatile u32* swCtrlLedReg;
static u32 ledFlashFrequency;

#define dvlEthLedStartSoftwareBlink() { blinkTimerState = 2; *swCtrlLedReg = E_COLLISION; mod_timer(&dvlEthLedBlinkTimer, jiffies + cLedActiveJiffies); }

inline struct vlan_ethhdr* dvlVlanEthHdr(const struct sk_buff* skb)
{
	return (struct vlan_ethhdr *)skb->data;
}

inline bool dvlEthLedCheckAndUpdatePacketCounter(void)
{
	bool rv = false;
	int i;
	unsigned short cnt;

	for (i = 0; i < nPortsInUse; i++)
	{
		cnt = (*(pktCntAdr[i])) & PC_MASK;
		if (cnt != pktCnt[i]) { rv = true; pktCnt[i] = cnt; }
	}
	return rv;
}

static void dvlEthLedSwPktCntPollTimerHandler(unsigned long unused)
{
	if (swCtrlLedReg)
	{
		if (blinkTimerState) mod_timer(&dvlEthLedSwPktCntPollTimer, jiffies + cSwPktCntPollJiffies);
		else if (dvlEthLedCheckAndUpdatePacketCounter() && !cpuActivity)
		{
			switchActivity = true;
			dvlEthLedStartSoftwareBlink();
		}
		else
		{
			cpuActivity = false;
			mod_timer(&dvlEthLedSwPktCntPollTimer, jiffies + cSwPktCntPollJiffies);
		}
	}
}

static void dvlEthLedBlinkTimerHandler(unsigned long unused)
{
	blinkTimerState--;
	if (blinkTimerState == 1)
	{
		if (swCtrlLedReg) *swCtrlLedReg = E_LINK;
		mod_timer(&dvlEthLedBlinkTimer, jiffies + cLedInactiveJiffies);
	}
	else if (switchActivity && swCtrlLedReg)
	{
		if (!dvlEthLedCheckAndUpdatePacketCounter())
		{
			switchActivity = false;
			mod_timer(&dvlEthLedSwPktCntPollTimer, jiffies + cSwPktCntPollJiffies);
		}
		else dvlEthLedStartSoftwareBlink();
	}
}

static void dvlEthLedRelease(void)
{
	int i;

	for (i = 0; i < sizeof(portsInUse); i++) _ESW_REG(R_P0_LED + (portsInUse[i]<<2)) = E_LINK_ACTIVITY;
}

static void dvlEthLedSetup(void)
{
	const unsigned int cLinks = _ESW_REG(R_POA) >> B_P0_LINK;
	unsigned int links, noLinks;
	int i;

	swCtrlLedReg = NULL;
	for (i = 0, noLinks = 0, links = cLinks; i < sizeof(portsInUse); i++)
	{
		links >>= portsInUse[i] - (i ? portsInUse[i-1] : 0);
		noLinks += links & 0x01;
	}
	switch (noLinks)
	{
		case 0:
		case 1:
			dvlEthLedRelease();
			break;
		default:
			for (i = 0, links = cLinks; i < sizeof(portsInUse); i++)
			{
				links >>= portsInUse[i] - (i ? portsInUse[i-1] : 0); 
				if (links & 0x01)
				{
					if (!swCtrlLedReg)
					{
						swCtrlLedReg = (u32 *)(RALINK_ETH_SW_BASE + R_P0_LED + (portsInUse[i]<<2));
						_ESW_REG(R_P0_LED + (portsInUse[i]<<2)) = E_LINK;
					}
					else _ESW_REG(R_P0_LED + (portsInUse[i]<<2)) = E_COLLISION;
				}
				else _ESW_REG(R_P0_LED + (portsInUse[i]<<2)) = E_LINK;
			}
			dvlEthLedCheckAndUpdatePacketCounter();
			mod_timer(&dvlEthLedSwPktCntPollTimer, jiffies + cSwPktCntPollJiffies);
			break;
	}
}

static void dvlEthLedLnkStatChg(unsigned long unused)
{
	dvlEthLedSetup();
}

void dvlEthLedInit(void)
{
	int i;

	swCtrlLedReg = NULL;
	for (i = 0; i < nPortsInUse; i++) pktCntAdr[i] = (u32 *)(RALINK_ETH_SW_BASE + R_P0PC + (portsInUse[i]<<2));
	switchActivity = false;
	cpuActivity = false;
	blinkTimerState = 0;
	ledFlashFrequency = _ESW_REG(R_SGC) & LED_FLASH_TIME_MASK;
	_ESW_REG(R_SGC) = (_ESW_REG(R_SGC) & (~LED_FLASH_TIME_MASK)) | (E_30MS << B_LED_FLASH_TIME);
	init_timer(&dvlEthLedBlinkTimer);
	dvlEthLedBlinkTimer.function = dvlEthLedBlinkTimerHandler;
	dvlEthLedBlinkTimer.data = 0;
	init_timer(&dvlEthLedSwPktCntPollTimer);
	dvlEthLedSwPktCntPollTimer.function = dvlEthLedSwPktCntPollTimerHandler;
	dvlEthLedSwPktCntPollTimer.data = 0;
	tasklet_init(&dvl_link_tasklet, dvlEthLedLnkStatChg, 0);
	dvlEthLedSetup();
}

void dvlEthLedExit(void)
{
	del_timer_sync(&dvlEthLedSwPktCntPollTimer);
	blinkTimerState = 0;
	del_timer_sync(&dvlEthLedBlinkTimer);
	switchActivity = false;
	cpuActivity = false;
	tasklet_kill(&dvl_link_tasklet);
	swCtrlLedReg = NULL;
	dvlEthLedRelease();
	_ESW_REG(R_SGC) = (_ESW_REG(R_SGC) & (~LED_FLASH_TIME_MASK)) | ledFlashFrequency;
}


void dvlEthLedPktTx(struct sk_buff *skb)
{
	if (!blinkTimerState && swCtrlLedReg && __be16_to_cpu(dvlVlanEthHdr(skb)->h_vlan_proto) == ETH_P_8021Q && (__be16_to_cpu(dvlVlanEthHdr(skb)->h_vlan_TCI) & VLAN_VID_MASK) == 1)
	{
		cpuActivity = true;
		dvlEthLedStartSoftwareBlink();
	}
}

void dvlEthLedPktRx(struct sk_buff *skb)
{
	if (!blinkTimerState && swCtrlLedReg && __be16_to_cpu(vlan_eth_hdr(skb)->h_vlan_proto) == ETH_P_8021Q && (__be16_to_cpu(vlan_eth_hdr(skb)->h_vlan_TCI) & VLAN_VID_MASK) == 1)
	{
		cpuActivity = true;
		dvlEthLedStartSoftwareBlink();
	}
}
