#include <command.h>
#include <common.h>
#include <rt_mmap.h>

extern int do_bootm (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);

#ifdef DVL_UBOOT_RESCUE_SUPPORT

extern flash_info_t flash_info[CFG_MAX_FLASH_BANKS];

DECLARE_GLOBAL_DATA_PTR;

#define RT2880_PIODATA_R (RALINK_PIO_BASE + 0X20)
#define RT2880_SYSCTLTESTSTAT2_R (RALINK_SYSCTL_BASE + 0X1C)

#define MTDPARTS_MAX_SIZE 256

#define RESC_FULL_HEADER_MAGIC 0xFB221172

#define VERSION_SIZE 128
typedef struct {
    unsigned int magic;
    unsigned int size;
    unsigned int chksum;
    unsigned int reserved;
    unsigned int fsOffset;
    unsigned char version[VERSION_SIZE];
/*  Total size must be 148 (0x94) bytes. 37 (0x25) unsigned ints. */
} DvlRescFullHeader;

typedef struct dvl_flash_info_struct
{
    ulong startAddr;
    ulong endAddr;
    unsigned int startSect;
    unsigned int endSect;
    ulong rescImgAddr;
    ulong fullImgAddr;
} DvlFlashInfo;

typedef struct mtd_partition_struct
{
    char* name;
    unsigned short offset;
    unsigned short size;
} MtdPartition;

static void dvlChksumUpdate(unsigned int *sum, unsigned char *buf, unsigned int size)
{
    register unsigned int chksum = 0;
    while(size--) chksum += *(buf++);
    *sum += chksum;
}

static void dvlChksumReverse(unsigned int *sum, unsigned char *buf, unsigned int size)
{
    register unsigned int chksum = 0;
    while(size--) chksum -= *(buf++);
    *sum += chksum;
}

static int dvlIsFlashSectorAligned(ulong addr)
{
    int i;
    if (!addr) return 0;
    for (i = 0; i < CFG_MAX_FLASH_SECT; i++)
    {
        if (addr == flash_info[0].start[i]) return 1;
        else if (!flash_info[0].start[i]) break;
    }
    return 0;
}

static ulong dvlGetFlashSectorEnd(DvlFlashInfo *dvlFlashInfo, ulong addr)
{
    int i;

    if (!addr) return 0;
    for (i = 0; i < CFG_MAX_FLASH_SECT; i++)
    {
        long cAddr = flash_info[0].start[i];
        if (!cAddr) break;
        if (cAddr >= addr) return cAddr;
    }
    return dvlFlashInfo->endAddr;
}

static int dvlCheckImageIntegrity(DvlFlashInfo* dvlFlashInfo, DvlRescFullHeader* image)
{
    unsigned int checksum = 0;
    if (ntohl(image->magic) != RESC_FULL_HEADER_MAGIC) return 0;
    if (ntohl(image->size) == 0) return 0;
    if (dvlFlashInfo->endAddr-(ulong)image < ntohl(image->size)+sizeof(DvlRescFullHeader)) return 0;
    dvlChksumUpdate(&checksum, (unsigned char *)image, ntohl(image->size)+sizeof(DvlRescFullHeader));
    dvlChksumReverse(&checksum, (unsigned char *)&(image->chksum), sizeof(unsigned int));
    if (checksum != ntohl(image->chksum)) return 0;
    if (!strlen(image->version)) return 0;
    if (strlen(image->version) >= VERSION_SIZE) return 0;
    if (ntohl(image->fsOffset) == 0) return 0;
    if (ntohl(image->fsOffset) >= ntohl(image->size)+sizeof(DvlRescFullHeader)) return 0;
    return dvlIsFlashSectorAligned((ulong)image + ntohl(image->fsOffset));
}

static ulong dvlFindImage(DvlFlashInfo* dvlFlashInfo, ulong ignoreUpToAddr)
{
    DvlRescFullHeader* imageHdr = NULL;
    unsigned int cSect;

    for (cSect = dvlFlashInfo->startSect; cSect < dvlFlashInfo->endSect && flash_info[0].start[cSect] <= ignoreUpToAddr; cSect++) {}
    for (; !imageHdr && cSect < dvlFlashInfo->endSect; cSect++)
    {
        DvlRescFullHeader* flashPtr = (DvlRescFullHeader *)flash_info[0].start[cSect];
        if (dvlCheckImageIntegrity(dvlFlashInfo, flashPtr)) imageHdr = flashPtr;
    }
    return (ulong)imageHdr;
}

static char* dvlGetMtdparts(DvlFlashInfo* info, int forceRescue);

int dvlBoot(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[], int forceRescue)
{
    bd_t *bd = gd->bd;
    DvlFlashInfo dvl_flash_info;
    unsigned int i;
    char* mtdparts;
    long imageAddr;

    /* check for available kernel start address */
    if (argc < 2) goto fallback;

    /* check for single flash bank */
    if (flash_info[0].size != bd->bi_flashsize) goto fallback;

    memset(&dvl_flash_info, 0, sizeof(DvlFlashInfo));

    /* get flash start and end addresses */
    dvl_flash_info.startAddr = simple_strtoul(argv[1], NULL, 16);
    if (!dvl_flash_info.startAddr || dvl_flash_info.startAddr < bd->bi_flashstart) goto fallback;
    dvl_flash_info.endAddr = bd->bi_flashstart+bd->bi_flashsize;
    if (dvl_flash_info.endAddr < dvl_flash_info.startAddr) goto fallback;

    /* check for uboot header @ begin of flash */
    if (ntohl(((image_header_t *)dvl_flash_info.startAddr)->ih_magic) == IH_MAGIC) goto fallback;

    /* get flash start and end sectors */
    for (i = 0; i < CFG_MAX_FLASH_SECT; i++)
    {
        if (flash_info[0].start[i] == dvl_flash_info.startAddr) { dvl_flash_info.startSect = i; break; }
        if (!flash_info[0].start[i] || (i && flash_info[0].start[i] <= flash_info[0].start[i-1])) break; /* something's fishy */
    }
    for (i = dvl_flash_info.startSect; i < CFG_MAX_FLASH_SECT; i++)
    {
        if (!flash_info[0].start[i] || flash_info[0].start[i] == dvl_flash_info.endAddr) { dvl_flash_info.endSect = i; break; }
        if (flash_info[0].start[i] <= flash_info[0].start[i-1] || flash_info[0].start[i] > dvl_flash_info.endAddr) break; /* something's fishy */
    }
    if (!dvl_flash_info.startSect || !dvl_flash_info.endSect) goto fallback;

    /* find images in flash */
    imageAddr = dvlFindImage(&dvl_flash_info, 0);
    if (!imageAddr) goto fallback;
    if (imageAddr == dvl_flash_info.startAddr)
    {
        dvl_flash_info.fullImgAddr = imageAddr;
        dvl_flash_info.rescImgAddr = dvlFindImage(&dvl_flash_info, imageAddr);
    }
    else dvl_flash_info.rescImgAddr = imageAddr;

    /* perform sanity checks and generate linux mtdparts cmdline parameter */
    mtdparts = dvlGetMtdparts(&dvl_flash_info, forceRescue);
    if (!strlen(mtdparts)) goto fallback;

    setenv("bootargs", mtdparts);

    if (!dvl_flash_info.fullImgAddr || forceRescue) sprintf(argv[1], "0x%lX", dvl_flash_info.rescImgAddr+sizeof(DvlRescFullHeader));
    else sprintf(argv[1], "0x%lX", dvl_flash_info.fullImgAddr+sizeof(DvlRescFullHeader));

fallback:
    return do_bootm(cmdtp, flag, argc, argv);
}

static char* dvlGetMtdparts(DvlFlashInfo* dvlFlashInfo, int forceRescue)
{
    int i;
    bd_t *bd = gd->bd;
    static char mtdparts[MTDPARTS_MAX_SIZE] = "";
    DvlRescFullHeader* resc_hdr = (DvlRescFullHeader *) dvlFlashInfo->rescImgAddr;
    DvlRescFullHeader* full_hdr = (DvlRescFullHeader *) dvlFlashInfo->fullImgAddr;
    DvlRescFullHeader* sing_hdr = NULL;
    unsigned short current_offset;
    MtdPartition mtd_partition[] = {
        { "Bootloader",   0, 128 },
        { "Config1"   , 128,  64 },
        { "Config2"   , 192,  64 },
        { "Kernel"    ,   0,   0 },
        { "RootFS"    ,   0,   0 },
        { "Rescue"    ,   0,   0 },
        { "Firmware"  ,   0,   0 },
        { "UpdResc"   ,   0,   0 },
        { "UpdFirm"   ,   0,   0 },
        { "Firmware1" , 256, ((dvlFlashInfo->endAddr - bd->bi_flashstart) >> 10) - 256 }
    };

    if (resc_hdr && !full_hdr) sing_hdr = resc_hdr;
    if (!resc_hdr && full_hdr) sing_hdr = full_hdr;

    /* sanity checks */
    if (sing_hdr)
    {
        if ((ulong)sing_hdr != dvlFlashInfo->startAddr && dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)sing_hdr+ntohl(sing_hdr->size)+sizeof(DvlRescFullHeader)) != dvlFlashInfo->endAddr) goto end;
    }
    else if (resc_hdr && full_hdr)
    {
        if (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)full_hdr+ntohl(full_hdr->size)+sizeof(DvlRescFullHeader)) > (ulong)resc_hdr) goto end;
        if (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)resc_hdr+ntohl(resc_hdr->size)+sizeof(DvlRescFullHeader)) != dvlFlashInfo->endAddr) goto end;
    }
    else goto end;

    /* generate partition data for Kernel and RootFS partitions */  
    if (!full_hdr || (resc_hdr && forceRescue))
    {
        mtd_partition[3].offset = ((ulong)resc_hdr - bd->bi_flashstart) >> 10;
        mtd_partition[3].size = ntohl(resc_hdr->fsOffset) >> 10;
        mtd_partition[4].offset = mtd_partition[3].offset + mtd_partition[3].size;
        mtd_partition[4].size = (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)resc_hdr+ntohl(resc_hdr->size)+sizeof(DvlRescFullHeader)) - (ulong)resc_hdr - ntohl(resc_hdr->fsOffset)) >> 10;
    }
    else
    {
        mtd_partition[3].offset = ((ulong)full_hdr - bd->bi_flashstart) >> 10;
        mtd_partition[3].size = ntohl(full_hdr->fsOffset) >> 10;
        mtd_partition[4].offset = mtd_partition[3].offset + mtd_partition[3].size;
        mtd_partition[4].size = (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)full_hdr+ntohl(full_hdr->size)+sizeof(DvlRescFullHeader)) - (ulong)full_hdr - ntohl(full_hdr->fsOffset)) >> 10;
    }

    /* generate partition data for Rescue and Firmware partitions */  
    if (resc_hdr)
    {
        mtd_partition[5].offset = ((ulong)resc_hdr - bd->bi_flashstart) >> 10;
        mtd_partition[5].size = (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)resc_hdr+ntohl(resc_hdr->size)+sizeof(DvlRescFullHeader)) - (ulong)resc_hdr) >> 10;
    }
    if (full_hdr)
    {
        mtd_partition[6].offset = ((ulong)full_hdr - bd->bi_flashstart) >> 10;
        mtd_partition[6].size = (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)full_hdr+ntohl(full_hdr->size)+sizeof(DvlRescFullHeader)) - (ulong)full_hdr) >> 10;
    }

    /* generate partition data for UpdResc and UpdFirm partitions */ 
    if (full_hdr)
    {
        mtd_partition[7].offset = (dvlGetFlashSectorEnd(dvlFlashInfo, (ulong)full_hdr+ntohl(full_hdr->size)+sizeof(DvlRescFullHeader)) - bd->bi_flashstart) >> 10;
        mtd_partition[7].size = ((dvlFlashInfo->endAddr - bd->bi_flashstart) >> 10) - mtd_partition[7].offset;
    }
    if (resc_hdr)
    {
        mtd_partition[8].offset = (dvlFlashInfo->startAddr - bd->bi_flashstart) >> 10;
        mtd_partition[8].size = (((ulong)resc_hdr - bd->bi_flashstart) >> 10) - mtd_partition[8].offset;
    }

    /* build mtdparts string */
    strcpy(mtdparts, "mtdparts=rt3052");
    for (current_offset = 0, i = 0; i < sizeof(mtd_partition) / sizeof(mtd_partition[0]); i++)
    {
        if (mtd_partition[i].size)
        {
            size_t ssize = strlen(mtdparts);
            /* snprintf is not available in uboot */
            if (MTDPARTS_MAX_SIZE - ssize > 16 + strlen(mtd_partition[i].name))
            {
                sprintf(mtdparts+ssize, "%c", i ? ',' : ':');
                if (current_offset == mtd_partition[i].offset) sprintf(mtdparts+ssize + 1, "%uk(%s)", mtd_partition[i].size, mtd_partition[i].name);
                else sprintf(mtdparts+ssize + 1, "%uk@%uk(%s)", mtd_partition[i].size, mtd_partition[i].offset, mtd_partition[i].name);
                current_offset = mtd_partition[i].offset + mtd_partition[i].size;
            }
            else { mtdparts[0] = 0; break; }
        }
    }
end:
    return mtdparts;
}

int dvlRescueFirmwareSelect(void)
{
    int rv = 1;
    if (le32_to_cpu(*(volatile u_long *)RT2880_SYSCTLTESTSTAT2_R) || le32_to_cpu(*(volatile u_long *)RT2880_PIODATA_R) & 0x400) rv = 0;
    *((volatile u32 *)(RT2880_SYSCTLTESTSTAT2_R)) = 0x2211;
    return rv;
}

#else /* DVL_UBOOT_RESCUE_SUPPORT */

int dvlRescueFirmwareSelect(void)
{
    return 0;
}

int dvlBoot(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[], int forceRescue)
{
    return do_bootm(cmdtp, flag, argc, argv);
}

#endif /* DVL_UBOOT_RESCUE_SUPPORT */
