/*
 * ledtrig-gio.c - LED Trigger Based on GPIO events
 *
 * Copyright 2009 Felipe Balbi <me@felipebalbi.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include "ledtrig-gpio.h"

struct gpio_trig_data {
	struct led_classdev *led;
	struct work_struct work;

	unsigned desired_brightness;	/* desired brightness when led is on */
	unsigned inverted;		/* true when gpio is inverted */
	unsigned gpio;			/* gpio that triggers the leds */
};

static irqreturn_t gpio_trig_irq(int irq, void *_led)
{
	struct led_classdev *led = _led;
	struct gpio_trig_data *gpio_data = led->trigger_data;

	/* just schedule_work since gpio_get_value can sleep */
	schedule_work(&gpio_data->work);

	return IRQ_HANDLED;
};

static void gpio_trig_work(struct work_struct *work)
{
	struct gpio_trig_data *gpio_data = container_of(work,
			struct gpio_trig_data, work);
	int tmp;

	if (!gpio_data->gpio)
		return;

	tmp = gpio_get_value(gpio_data->gpio);
	if (gpio_data->inverted)
		tmp = !tmp;

	if (tmp) {
		if (gpio_data->desired_brightness)
			dvlled_brightness_set(gpio_data->led,
					   gpio_data->desired_brightness);
		else
			dvlled_brightness_set(gpio_data->led, LED_FULL);
	} else {
		dvlled_brightness_set(gpio_data->led, LED_OFF);
	}
}

static ssize_t gpio_trig_brightness_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct gpio_trig_data *gpio_data = led->trigger_data;

	return sprintf(buf, "%u\n", gpio_data->desired_brightness);
}

static ssize_t gpio_trig_brightness_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct gpio_trig_data *gpio_data = led->trigger_data;
	unsigned desired_brightness;
	int ret;

	ret = sscanf(buf, "%u", &desired_brightness);
	if (ret < 1 || desired_brightness > 255) {
		dev_err(dev, "invalid value\n");
		return -EINVAL;
	}

	gpio_data->desired_brightness = desired_brightness;

	return n;
}
static DEVICE_ATTR(desired_brightness, 0644, gpio_trig_brightness_show,
		gpio_trig_brightness_store);

static ssize_t request_mirrored_gpio_irq(struct led_classdev *led, const int gpio)
{
	int ret;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,33)
    ret = request_threaded_irq(
            gpio_to_irq(gpio), 
            NULL,
            gpio_trig_irq,
#else
    ret = request_irq(
            gpio_to_irq(gpio), 
            gpio_trig_irq,
#endif
            IRQF_SHARED | IRQ_TYPE_EDGE_BOTH,
            "ledtrig-gpio", 
            led);

	if (ret) {
		printk(KERN_ERR "dvlled: request_irq failed with error %d\n", ret);
	} else {
    	printk(KERN_INFO "dvlled: IRQ %d requested, %s\n", gpio_to_irq(gpio), ret < 0 ? "failed" : "successful");
	}

	return ret ? ret : 0;
}

static void gpio_trig_activate(struct led_classdev *led)
{
	struct gpio_trig_data *gpio_data;
	struct dvlled_entry *dvlLedConfig;

	int ret, i;

	gpio_data = kzalloc(sizeof(*gpio_data), GFP_KERNEL);
	if (!gpio_data)
		return;

    //make dvl led config accessable
	dvlLedConfig = &global_leds;

    //configure mirrored gpio and inverted state
	for (i = 0; i < num_leds(); i++)
	{
        //Note: parameter led contains the current led
        //      dvlLedConfig the config of all dvl known leds

        //skip all not desired leds
        if (strcmp(dvlLedConfig[i].name,led->name))
        {
            //printk("skip %s, this is %s\n",dvlLedConfig[i].name,led->name);
            continue;
        }

        //set led and request gpio
        gpio_data->gpio = dvlLedConfig[i].line2;
        ret = request_mirrored_gpio_irq(led,gpio_data->gpio);
        if (ret)
            goto err;

        //set inverted
        if (dvlLedConfig[i].polarity_in == POL_LA)
            gpio_data->inverted = 1;

        //party done
        break;

    }

	ret = device_create_file(led->dev, &dev_attr_desired_brightness);
	if (ret)
		goto err;

	gpio_data->led = led;
	led->trigger_data = gpio_data;
	INIT_WORK(&gpio_data->work, gpio_trig_work);

	return;

err:
	kfree(gpio_data);
}

static void gpio_trig_deactivate(struct led_classdev *led)
{
	struct gpio_trig_data *gpio_data = led->trigger_data;

	if (gpio_data) {
		device_remove_file(led->dev, &dev_attr_desired_brightness);
		flush_work(&gpio_data->work);
		if (gpio_data->gpio != 0)
			free_irq(gpio_to_irq(gpio_data->gpio), led);
		kfree(gpio_data);
        led->trigger_data = NULL;
	}
}

struct led_trigger gpio_led_trigger = {
	.name		= "gpio",
	.activate	= gpio_trig_activate,
	.deactivate	= gpio_trig_deactivate,
};
